<?php
/*
  CAMBIO respecto a los scripts anteriores:
  - Antes: Imagick o ejecutable "magick/convert".
  - Ahora: SOLO PHP con FPDF (sin dependencias externas del sistema).

  Qué hace:
  - Recorre un array de carpetas.
  - Junta todos los .jpg/.jpeg ordenados naturalmente (000.jpg, 001.jpg, ...).
  - Crea un PDF por carpeta dentro de la misma carpeta.
*/

error_reporting(E_ALL);
ini_set("display_errors", 1);

/* --------------------- CONFIGURACIÓN --------------------- */
// Pasa aquí tus carpetas (rutas absolutas o relativas)
$carpetas = [
  // Ejemplos:
  __DIR__ . DIRECTORY_SEPARATOR . "S1NLA",
  __DIR__ . DIRECTORY_SEPARATOR . "S1INA",
  __DIR__ . DIRECTORY_SEPARATOR . "S1LEA",
];

/* --------------------- VERIFICAR FPDF --------------------- */
// Comentario: verificamos que exista fpdf.php en el mismo directorio del script o en el include_path.
if (!class_exists("FPDF")) {
  // Intentamos cargar fpdf.php desde el mismo directorio del script
  $rutaFPDFLocal = __DIR__ . DIRECTORY_SEPARATOR . "fpdf.php";
  if (file_exists($rutaFPDFLocal)) {
    require_once $rutaFPDFLocal;
  } else {
    // Si no está, intentamos un require genérico por si el usuario lo agregó al include_path
    // Si falla, mostramos mensaje claro.
    @require_once "fpdf.php";
    if (!class_exists("FPDF")) {
      echo "No se encontró fpdf.php.\n";
      echo "Descarga FPDF (fpdf.php) y colócalo junto a este script o en el include_path.\n";
      exit(1);
    }
  }
}

/* --------------------- FUNCIÓN PRINCIPAL --------------------- */
function crearPDFConFPDF($carpeta) {
  // Comentario: normalizamos ruta y verificamos existencia
  $carpeta = rtrim($carpeta, DIRECTORY_SEPARATOR);
  if (!is_dir($carpeta)) {
    echo "[Aviso] Carpeta no existe: " . $carpeta . PHP_EOL;
    return false;
  }

  // Comentario: recolectamos imágenes .jpg/.jpeg y ordenamos naturalmente
  $archivos = array_merge(
    glob($carpeta . DIRECTORY_SEPARATOR . "*.jpg"),
    glob($carpeta . DIRECTORY_SEPARATOR . "*.jpeg")
  );
  natsort($archivos);
  $archivos = array_values($archivos);

  if (count($archivos) == 0) {
    echo "[Aviso] No hay imágenes JPG en: " . $carpeta . PHP_EOL;
    return false;
  }

  // Comentario: preparamos salida PDF (mismo nombre que la carpeta)
  $nombreCarpeta = basename($carpeta);
  $pdfSalida = $carpeta . DIRECTORY_SEPARATOR . $nombreCarpeta . ".pdf";

  // Comentario: instanciamos FPDF con métricas en mm y sin saltos automáticos
  $pdf = new FPDF("P", "mm", "A4");
  $pdf->SetAutoPageBreak(false);

  // Comentario: tamaños de página A4 en mm
  // FPDF proporciona métodos para leer ancho/alto de la página actual.
  // Usaremos AddPage con orientación dinámica por imagen.
  foreach ($archivos as $img) {
    // Comentario: obtenemos tamaño de la imagen para decidir orientación y escala
    $info = @getimagesize($img);
    if ($info === false) {
      // Comentario: si no se puede leer, la saltamos
      echo "[Aviso] No se pudo leer la imagen: " . $img . PHP_EOL;
      continue;
    }
    $wpx = $info[0];
    $hpx = $info[1];

    // Comentario: orientamos la página según la imagen
    $orient = ($wpx >= $hpx) ? "L" : "P";
    $pdf->AddPage($orient);

    // Comentario: dimensiones de página actuales en mm
    $pageW = $pdf->GetPageWidth();
    $pageH = $pdf->GetPageHeight();

    // Comentario: margen mínimo (mm). Lo ponemos a 0 para llenar al máximo.
    $margen = 0;
    $maxW = $pageW - 2 * $margen;
    $maxH = $pageH - 2 * $margen;

    // Comentario: ajustamos la imagen a la página manteniendo proporción
    $ratioImg = $wpx / $hpx;
    $ratioPag = $maxW / $maxH;

    if ($ratioImg >= $ratioPag) {
      // Comentario: imagen más “ancha”, ajustamos al ancho máximo
      $w = $maxW;
      $h = $w / $ratioImg;
    } else {
      // Comentario: imagen más “alta”, ajustamos al alto máximo
      $h = $maxH;
      $w = $h * $ratioImg;
    }

    // Comentario: centramos la imagen en la página
    $x = ($pageW - $w) / 2;
    $y = ($pageH - $h) / 2;

    // Comentario: colocamos la imagen. FPDF se encarga del escalado.
    $pdf->Image($img, $x, $y, $w, $h);
  }

  // Comentario: guardamos el PDF en disco
  $pdf->Output("F", $pdfSalida);

  if (file_exists($pdfSalida) && filesize($pdfSalida) > 0) {
    echo "[OK] PDF generado: " . $pdfSalida . PHP_EOL;
    return true;
  } else {
    echo "[Error] No se pudo generar el PDF en: " . $carpeta . PHP_EOL;
    return false;
  }
}

/* --------------------- PROCESO --------------------- */
// Comentario: recorremos cada carpeta y generamos su PDF
foreach ($carpetas as $c) {
  crearPDFConFPDF($c);
}

echo "Proceso finalizado.\n";
